const express = require('express');
const app = express();
const path = require('path');

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));
app.use(express.static(path.join(__dirname, 'public')));
app.use(express.urlencoded({ extended: true }));

let items = [
  { id: "1", name: "Milk", quantity: "2" },
  { id: "2", name: "Bread", quantity: "1" }
];

function validateItem(name, quantity) {
  const errors = {};
  const trimmed = name.trim();

  if (!trimmed) errors.name = "Item name is required.";
  else if (trimmed.length < 2) errors.name = "Item name must be at least 2 characters.";

  if (!quantity) errors.quantity = "Quantity is required.";
  else if (Number(quantity) <= 0 || !Number.isInteger(Number(quantity)))
    errors.quantity = "Quantity must be a positive integer.";

  return errors;
}

app.get('/', (req, res) => {
  res.render('index', { items, formErrors: {}, formData: {} });
});

app.post('/items', (req, res) => {
  const { name, quantity } = req.body;
  const errors = validateItem(name, quantity);

  if (Object.keys(errors).length > 0) {
    return res.status(400).render('index', {
      items,
      formErrors: errors,
      formData: { name, quantity }
    });
  }

  items.push({
    id: Date.now().toString(),
    name: name.trim(),
    quantity: quantity.trim()
  });

  res.redirect('/');
});

app.get('/items/:id/edit', (req, res) => {
  const item = items.find(i => i.id === req.params.id);
  if (!item) return res.status(404).send("Item not found");
  res.render('edit', { item, formErrors: {} });
});

app.post('/items/:id', (req, res) => {
  const { name, quantity } = req.body;
  const errors = validateItem(name, quantity);
  const item = items.find(i => i.id === req.params.id);

  if (!item) return res.status(404).send("Item not found");

  if (Object.keys(errors).length > 0) {
    return res.render('edit', {
      item: { ...item, name, quantity },
      formErrors: errors
    });
  }

  item.name = name.trim();
  item.quantity = quantity.trim();
  res.redirect('/');
});

app.get('/items/:id/delete', (req, res) => {
  items = items.filter(i => i.id !== req.params.id);
  res.redirect('/');
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => console.log(`Server running on port ${PORT}`));
